<?php
/**
 * Print Agent Pro 3.0 – Server Core
 * Features:
 *  - JWT Login (users.json)
 *  - Roles: admin/operator/viewer
 *  - add_job / get_jobs / update_job / analytics / diagnostics / list_users / add_user
 *  - Third-party publish (Pusher/Ably) if ENV set
 *  - E2E content agnostic storage
 *  - Fallback order: Node WS → PHP WS → Third-Party → Polling
 */

define('SECRET_TOKEN','CHANGE_ME_TOKEN');
define('JOBS_FILE', __DIR__.'/jobs.json');
define('ANALYTICS_FILE', __DIR__.'/analytics.json');
define('USERS_FILE', __DIR__.'/users.json');
define('SERVER_LOG', __DIR__.'/server.log');
define('NODE_WS_PORT',7071);
define('PHP_WS_PORT',8080);
define('JWT_SECRET','CHANGE_ME_LONG_RANDOM');
define('JWT_EXPIRE',3600);

session_start();

/* ---------- Generic Helpers ---------- */
function json_response($arr){ header('Content-Type:application/json; charset=utf-8'); echo json_encode($arr,JSON_UNESCAPED_UNICODE); exit; }
function safe_json_read($file,$default){
  if(!file_exists($file)) file_put_contents($file,json_encode($default,JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
  $raw=@file_get_contents($file);
  $j=json_decode($raw,true);
  return is_array($j)?$j:$default;
}
function safe_json_write($file,$data){
  $fp=fopen($file,'w'); flock($fp,LOCK_EX); fwrite($fp,json_encode($data,JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE)); flock($fp,LOCK_UN); fclose($fp);
}
function read_jobs(){ return safe_json_read(JOBS_FILE,[]); }
function write_jobs($j){ safe_json_write(JOBS_FILE,$j); }
function read_analytics(){ return safe_json_read(ANALYTICS_FILE,["totals"=>["pending"=>0,"printed"=>0,"failed"=>0],"by_type"=>[],"daily"=>[]]); }
function write_analytics($a){ safe_json_write(ANALYTICS_FILE,$a); }
function read_users(){ return safe_json_read(USERS_FILE,[]); }
function log_server($msg){ file_put_contents(SERVER_LOG,'['.date('c').'] '.$msg.PHP_EOL,FILE_APPEND); }

function base64url($d){return rtrim(strtr(base64_encode($d),'+/','-_'),'=');}
function make_jwt($payload){
  $header=base64url(json_encode(['alg'=>'HS256','typ'=>'JWT']));
  $body=base64url(json_encode($payload));
  $sig=base64url(hash_hmac('sha256',"$header.$body",JWT_SECRET,true));
  return "$header.$body.$sig";
}
function parse_jwt($jwt){
  $parts=explode('.',$jwt); if(count($parts)!=3) return false;
  [$h,$b,$s]=$parts;
  $check=base64url(hash_hmac('sha256',"$h.$b",JWT_SECRET,true));
  if(!hash_equals($check,$s)) return false;
  $payload=json_decode(base64_decode(strtr($b,'-_','+/')),true);
  if(!$payload) return false;
  if(isset($payload['exp']) && time()>$payload['exp']) return false;
  return $payload;
}
function current_role($token){
  if(!$token) return null;
  if($token===SECRET_TOKEN) return 'admin'; // legacy
  $p=parse_jwt($token);
  return $p['role']??null;
}
function is_port_open($port){
  $c=@fsockopen('127.0.0.1',$port,$e,$s,0.2);
  if($c){fclose($c);return true;}
  return false;
}
function node_available(){
  if(!function_exists('shell_exec')) return false;
  $v=@shell_exec('node -v 2>/dev/null');
  if(!$v) return false;
  return preg_match('/^v\\d+\\.\\d+\\.\\d+/',trim($v))?trim($v):false;
}
function php_ws_possible(){ return extension_loaded('sockets') && extension_loaded('pcntl'); }
function publish_third_party($job){
  if(getenv('PUSHER_APP_ID') && getenv('PUSHER_KEY') && getenv('PUSHER_SECRET')){
    $appId=getenv('PUSHER_APP_ID'); $key=getenv('PUSHER_KEY'); $secret=getenv('PUSHER_SECRET');
    $cluster=getenv('PUSHER_CLUSTER')?:'mt1';
    $channel=getenv('PUSHER_CHANNEL')?:'print-jobs';
    $event='new_job';
    $payload=json_encode($job,JSON_UNESCAPED_UNICODE);
    $ts=time();
    $stringToSign="POST\n/apps/{$appId}/events\nauth_key={$key}&auth_timestamp={$ts}&auth_version=1.0&body_md5=".md5($payload);
    $sig=hash_hmac('sha256',$stringToSign,$secret);
    $url="https://api-{$cluster}.pusher.com/apps/{$appId}/events?auth_key={$key}&auth_timestamp={$ts}&auth_version=1.0&body_md5=".md5($payload)."&auth_signature=$sig";
    $ch=curl_init($url);
    curl_setopt_array($ch,[CURLOPT_POST=>true,CURLOPT_POSTFIELDS=>json_encode(['name'=>$event,'channel'=>$channel,'data'=>$payload]),CURLOPT_HTTPHEADER=>['Content-Type: application/json'],CURLOPT_RETURNTRANSFER=>true]);
    curl_exec($ch); curl_close($ch);
  }
  if(getenv('ABLY_KEY')){
    $channel=getenv('ABLY_CHANNEL')?:'print-jobs';
    $key=getenv('ABLY_KEY');
    $payload=json_encode(['name'=>'new_job','data'=>$job],JSON_UNESCAPED_UNICODE);
    $ch=curl_init("https://rest.ably.io/channels/{$channel}/messages");
    curl_setopt_array($ch,[CURLOPT_POST=>true,CURLOPT_POSTFIELDS=>$payload,CURLOPT_HTTPHEADER=>['Content-Type: application/json','Authorization: Basic '.base64_encode($key)],CURLOPT_RETURNTRANSFER=>true]);
    curl_exec($ch); curl_close($ch);
  }
}
function notify_ws($event,$data){
  $payload=json_encode(['event'=>$event,'data'=>$data],JSON_UNESCAPED_UNICODE);
  if(is_port_open(NODE_WS_PORT)) send_ws(NODE_WS_PORT,$payload);
  elseif(is_port_open(PHP_WS_PORT)) send_ws(PHP_WS_PORT,$payload);
  if($event==='new_job') publish_third_party($data);
}
function send_ws($port,$payload){
  $fp=@fsockopen('127.0.0.1',$port,$e,$s,0.2); if(!$fp)return;
  $key=base64_encode(random_bytes(16));
  $hdr="GET / HTTP/1.1\r\nHost:127.0.0.1:$port\r\nUpgrade:websocket\r\nConnection:Upgrade\r\nSec-WebSocket-Key:$key\r\nSec-WebSocket-Version:13\r\n\r\n";
  fwrite($fp,$hdr); $res=fread($fp,1024);
  if(strpos($res,' 101 ')===false){ fclose($fp); return; }
  $len=strlen($payload); if($len>125){ fclose($fp); return; }
  $frame=chr(0x81).chr($len).$payload;
  fwrite($fp,$frame); fclose($fp);
}
function analytics_add($job){
  $a=read_analytics();
  $a['totals']['pending']=($a['totals']['pending']??0)+1;
  $a['by_type'][$job['type']]=($a['by_type'][$job['type']]??0)+1;
  $day=date('Y-m-d');
  $a['daily'][$day]=$a['daily'][$day]??['added'=>0,'printed'=>0,'failed'=>0];
  $a['daily'][$day]['added']++;
  write_analytics($a);
}
function analytics_status($status){
  $a=read_analytics(); $day=date('Y-m-d');
  $a['daily'][$day]=$a['daily'][$day]??['added'=>0,'printed'=>0,'failed'=>0];
  if($status==='printed'){
    $a['totals']['printed']=($a['totals']['printed']??0)+1;
    $a['totals']['pending']=max(0,($a['totals']['pending']??0)-1);
    $a['daily'][$day]['printed']++;
  } elseif($status==='failed'){
    $a['totals']['failed']=($a['totals']['failed']??0)+1;
    $a['totals']['pending']=max(0,($a['totals']['pending']??0)-1);
    $a['daily'][$day]['failed']++;
  }
  write_analytics($a);
}
function disabled_funcs(){
  $d=ini_get('disable_functions');
  return array_filter(array_map('trim',explode(',',$d)));
}
function outbound_http(){
  if(!function_exists('curl_init')) return false;
  $ch=curl_init('https://example.com/');
  curl_setopt_array($ch,[CURLOPT_RETURNTRANSFER=>true,CURLOPT_TIMEOUT=>2]);
  curl_exec($ch);
  $err=curl_errno($ch);
  curl_close($ch);
  return $err===0;
}
function convert_bytes($val){
  $val=trim($val); $last=strtolower($val[strlen($val)-1]); $num=(int)$val;
  switch($last){ case 'g':$num*=1024; case 'm':$num*=1024; case 'k':$num*=1024; }
  return $num;
}

/* ---------- Router ---------- */
$action=$_GET['action']??null;

/* LOGIN */
if($action==='login' && $_SERVER['REQUEST_METHOD']==='POST'){
  $data=json_decode(file_get_contents('php://input'),true);
  $users=read_users();
  $user=null;
  foreach($users as $u){ if($u['username']===($data['username']??'')){$user=$u;break;} }
  if(!$user || !password_verify($data['password']??'', $user['password_hash'])){
    json_response(['error'=>'Invalid credentials']);
  }
  $jwt=make_jwt(['sub'=>$user['username'],'role'=>$user['role'],'exp'=>time()+JWT_EXPIRE]);
  json_response(['token'=>$jwt,'role'=>$user['role'],'expires_in'=>JWT_EXPIRE]);
}

if($action){
  $token=$_GET['token']??'';
  $role=current_role($token);
  if(!$role){
    http_response_code(401);
    json_response(['error'=>'Unauthorized']);
  }

  if($action==='list_users'){
    if($role!=='admin') json_response(['error'=>'Forbidden']);
    json_response(['users'=>read_users()]);
  }
  if($action==='add_user' && $_SERVER['REQUEST_METHOD']==='POST'){
    if($role!=='admin') json_response(['error'=>'Forbidden']);
    $data=json_decode(file_get_contents('php://input'),true);
    $users=read_users();
    $users[]=[
      'username'=>$data['username'],
      'password_hash'=>password_hash($data['password'],PASSWORD_BCRYPT),
      'role'=>$data['role']??'viewer'
    ];
    safe_json_write(USERS_FILE,$users);
    json_response(['success'=>true]);
  }

  if($action==='diagnostics'){
    $disabled=disabled_funcs();
    $need=['shell_exec','exec','popen'];
    $funcStatus=[];
    foreach($need as $f){ $funcStatus[$f]=function_exists($f) && !in_array($f,$disabled); }
    $nodeVer=node_available();
    $phpPossible=php_ws_possible();
    $mem=ini_get('memory_limit');
    $memBytes=convert_bytes($mem);
    $writable=[
      'jobs.json'=>is_writable(JOBS_FILE),
      'analytics.json'=>is_writable(ANALYTICS_FILE) || is_writable(__DIR__),
      'server.log'=>is_writable(__DIR__)
    ];
    $third=[
      'pusher_config'=>(bool)(getenv('PUSHER_APP_ID')&&getenv('PUSHER_KEY')&&getenv('PUSHER_SECRET')),
      'ably_config'=>(bool)getenv('ABLY_KEY')
    ];
    json_response([
      'node'=>['version'=>$nodeVer,'ws_running'=>is_port_open(NODE_WS_PORT)],
      'php_ws'=>['possible'=>$phpPossible,'ws_running'=>is_port_open(PHP_WS_PORT)],
      'third_party'=>$third,
      'memory'=>['limit'=>$mem,'bytes'=>$memBytes,'recommended_ok'=>$memBytes>=256*1024*1024],
      'disabled_functions'=>$disabled,
      'needed_functions'=>$funcStatus,
      'writable'=>$writable,
      'outbound_http'=>outbound_http(),
      'fallback_priority'=>['node_ws','php_ws','third_party','polling'],
      'timestamp'=>date('c')
    ]);
  }

  if($action==='add_job' && $_SERVER['REQUEST_METHOD']==='POST'){
    if(!in_array($role,['admin','operator'])) json_response(['error'=>'Forbidden']);
    $data=json_decode(file_get_contents('php://input'),true);
    if(!isset($data['content'])) json_response(['error'=>'content required']);
    $jobs=read_jobs();
    $job=[
      'id'=>uniqid('job_'),
      'content'=>$data['content'],
      'type'=>$data['type']??'html',
      'status'=>'pending',
      'created_at'=>date('Y-m-d H:i:s'),
      'printed_at'=>null,
      'encrypted'=>!empty($data['encrypted']),
      'retries'=>0,
      'last_error'=>null
    ];
    $jobs[]=$job;
    write_jobs($jobs);
    analytics_add($job);
    notify_ws('new_job',$job);
    log_server("Added job {$job['id']}");
    json_response(['success'=>true,'job'=>$job]);
  }

  if($action==='get_jobs'){
    $jobs=read_jobs();
    $pending=array_values(array_filter($jobs,fn($j)=>$j['status']==='pending'));
    json_response($pending);
  }

  if($action==='update_job' && $_SERVER['REQUEST_METHOD']==='POST'){
    if(!in_array($role,['admin','operator'])) json_response(['error'=>'Forbidden']);
    $data=json_decode(file_get_contents('php://input'),true);
    if(!isset($data['id'],$data['status'])) json_response(['error'=>'id/status required']);
    $jobs=read_jobs();
    foreach($jobs as &$j){
      if($j['id']===$data['id']){
        $j['status']=$data['status'];
        if($data['status']==='printed') $j['printed_at']=date('Y-m-d H:i:s');
        if($data['status']==='failed') $j['last_error']=$data['error']??'failed';
      }
    }
    write_jobs($jobs);
    analytics_status($data['status']);
    notify_ws('job_updated',['id'=>$data['id'],'status'=>$data['status']]);
    log_server("Updated job {$data['id']} => {$data['status']}");
    json_response(['success'=>true]);
  }

  if($action==='analytics'){
    json_response(read_analytics());
  }

  if($action==='reset_analytics'){
    if($role!=='admin') json_response(['error'=>'Forbidden']);
    write_analytics(["totals"=>["pending"=>0,"printed"=>0,"failed"=>0],"by_type"=>[],"daily"=>[]]);
    json_response(['success'=>true]);
  }

  json_response(['error'=>'Unknown action']);
}

/* Minimal dashboard (legacy) */
if(isset($_POST['login']) && ($_POST['password']??'')==='admin'){
  $_SESSION['admin']=true;
}
if(!isset($_SESSION['admin'])){
  echo '<form method="post"><input type="password" name="password"><button name="login">Login (legacy)</button></form>';
  exit;
}
$jobs=read_jobs(); $analytics=read_analytics();
?>
<!DOCTYPE html>
<html lang="fa">
<head><meta charset="utf-8"><title>Server Dashboard</title>
<style>
body{font-family:tahoma;direction:rtl;background:#fafafa;padding:20px;}
table{border-collapse:collapse;width:100%;background:#fff;}
th,td{border:1px solid #ccc;padding:6px;font-size:12px;}
</style>
</head>
<body>
<h1>Server Dashboard (Legacy Minimal)</h1>
<p>Total Jobs: <?=count($jobs)?> | Printed: <?=$analytics['totals']['printed']?> | Failed: <?=$analytics['totals']['failed']?></p>
<table>
<tr><th>ID</th><th>Type</th><th>Status</th><th>Created</th><th>Printed</th><th>Enc</th></tr>
<?php foreach(array_slice(array_reverse($jobs),0,15) as $j): ?>
<tr>
<td><?=$j['id']?></td>
<td><?=$j['type']?></td>
<td><?=$j['status']?></td>
<td><?=$j['created_at']?></td>
<td><?=$j['printed_at']??'-'?></td>
<td><?=$j['encrypted']?'Yes':'No'?></td>
</tr>
<?php endforeach; ?>
</table>
<form method="post" action="?create_test=1&token=<?=SECRET_TOKEN?>">
<textarea name="c" rows="3" style="width:100%"><h1>Test</h1><p><?=date('H:i:s')?></p></textarea>
<label><input type="checkbox" name="enc" value="1"> encrypted (demo)</label>
<button>ایجاد تست</button>
</form>
<?php
if(isset($_GET['create_test']) && $_SERVER['REQUEST_METHOD']==='POST'){
  $jobs=read_jobs();
  $content=$_POST['c'];
  if(!empty($_POST['enc'])){
    $salt=bin2hex(random_bytes(16));
    $iv=bin2hex(random_bytes(16));
    $cipher=base64_encode($content); // Placeholder – رمزنگاری واقعی در کلاینت
    $content="$salt.$iv.$cipher";
  }
  $job=[
    'id'=>uniqid('job_'),
    'content'=>$content,
    'type'=>'html',
    'status'=>'pending',
    'created_at'=>date('Y-m-d H:i:s'),
    'printed_at'=>null,
    'encrypted'=>!empty($_POST['enc']),
    'retries'=>0,
    'last_error'=>null
  ];
  $jobs[]=$job;
  write_jobs($jobs);
  analytics_add($job);
  notify_ws('new_job',$job);
  header('Location: admin.php'); exit;
}
?>
</body></html>